----------------
title : C++ 레퍼런스 - iter_swap 함수 (<algorithm>)
cat_title : iter_swap
ref_title : iter_swap, std::iter_swap
path : /C++ Reference/algorithm
publish_date : 2019-06-30
----------------

##@ cpp-ref-start

#@ iter_swap

`<algorithm>` 에 정의됨

```cpp-formatted
template <class ForwardIt1, class ForwardIt2>
void iter_swap(ForwardIt1 a, ForwardIt2 b);

// C++ 20 이후 부터
template <class ForwardIt1, class ForwardIt2>
constexpr void iter_swap(ForwardIt1 a, ForwardIt2 b);
```

두 반복자가 가리키고 있는 원소들의 값을 서로 바꾼다(swap).

### 인자들

* `a, b` : 서로 바꿀 (swap) 두 반복자


### 리턴값

없음

### 구현 예시

```cpp-formatted
template <class ForwardIt1, class ForwardIt2>
constexpr void iter_swap(ForwardIt1 a,
                         ForwardIt2 b)  // C++20 부터 constexpr 이 붙음
{
  using std::swap;
  swap(*a, *b);
}
```

### 실행 예제

예를 들어서 `iter_swap` 을 사용하면 벡터의 두 원소의 위치를 서로 바꿀 수 있다.

```cpp-formatted
#include <algorithm>
#include <iostream>
#include <vector>

void print_vec(const std::vector<int>& v) {
  std::cout << "[";
  for (auto i : v) {
    std::cout << i << " ";
  }
  std::cout << "]" << std::endl;
}

int main() {
  std::vector<int> v = {1, 2, 3, 4, 5};
  print_vec(v);

  // 2 와 3 을 바꾼다.
  std::iter_swap(v.begin() + 1, v.begin() + 2);

  print_vec(v);
}
```

```exec
[1 2 3 4 5 ]
[1 3 2 4 5 ]
```

#### 다른 예제

원소들을 선택 정렬하는 함수를 `iter_swap` 을 통해 만들어본다.

```cpp-formatted
#include <algorithm>
#include <functional>
#include <iostream>
#include <random>
#include <vector>

template <class ForwardIt>
void selection_sort(ForwardIt begin, ForwardIt end) {
  for (ForwardIt i = begin; i != end; ++i)
    std::iter_swap(i, std::min_element(i, end));
}

int main() {
  std::random_device rd;
  std::mt19937 gen(rd());
  std::uniform_int_distribution<> dist(-10, 10);
  std::vector<int> v;
  generate_n(back_inserter(v), 20, bind(dist, gen));

  std::cout << "Before sort: ";
  for (auto e : v) std::cout << e << " ";

  selection_sort(v.begin(), v.end());

  std::cout << "\nAfter sort: ";
  for (auto e : v) std::cout << e << " ";
  std::cout << '\n';
}
```

```exec
Before sort: -7 6 2 4 -1 6 -9 -1 2 -5 10 -9 -5 -3 -5 -3 6 6 1 8
After sort: -9 -9 -7 -5 -5 -5 -3 -3 -1 -1 1 2 2 4 6 6 6 6 8 10
```

### 참고 자료

* `swap` : 두 객체의 값을 서로 바꾼다.
* `swap_ranges` : 두 범위 내의 원소들을 서로 바꾼다.